#ifndef RP2_MAG_H
#define RP2_MAG_H
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#define RP2_MAG_NUM_TRACKS      3
#define RP2_MAG_MAX_RAW_LEN     128
#define RP2_MAG_MAX_ENCODED_LEN 128
#define RP2_MAG_MAX_XML_LEN     512

typedef enum { HICO, MIDCO, LOWCO } Rp2MagCoercivity;

typedef enum { BPI75 = 75, BPI210 = 210 } Rp2MagBitsPerInch;

typedef enum {
    BPC1 = 1,
    BPC4 = 4,
    BPC5 = 5,
    BPC7 = 7,
    BPC8 = 8
} Rp2MagBitsPerChar;

typedef enum { NONE, ISO, JIS } Rp2MagEcodingStandard;

/**
 * @brief General Mag settings which will apply to all tracks.
 */
struct Rp2MagSettings {
    Rp2MagCoercivity      coercivity;
    Rp2MagEcodingStandard standard;
    bool                  verify;
};

/**
 * @brief Data for a single track.
 */
struct Rp2MagTrackData {
    char    raw[RP2_MAG_MAX_RAW_LEN];
    uint8_t encoded[RP2_MAG_MAX_ENCODED_LEN];
    size_t  encoded_len_bits;
    size_t  encoded_len_round_bytes;
};

/**
 * @brief Mag settings specific to a track.
 */
struct Rp2MagTrackSettings {
    Rp2MagBitsPerInch bpi;
    Rp2MagBitsPerChar bpc;
};

/**
 * @brief A single mag track.
 */
struct Rp2MagTrack {
    uint8_t                    track_number;
    struct Rp2MagTrackSettings settings;
    struct Rp2MagTrackData     data;
};

/**
 * @brief The main struct to encapsulate all mag data for the Rio Pro 2
 * platform.
 */
struct Rp2Mag {
    struct Rp2MagSettings settings;
    struct Rp2MagTrack    tracks[RP2_MAG_NUM_TRACKS];
};

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

/**
 * @brief Read the CUPS options string and extract any Mag data into the
 * provided Rp2Mag struct.
 *
 * @param rp2_mag Pointer to the Rp2Mag struct to populate.
 * @param options_string the options from the command line (every '-o')
 * concatenated (space separated) into a single string. This is usually argv[5]
 * of the cups filter.
 */
void rp2mag_parse_mag_command_line(struct Rp2Mag *rp2_mag,
                                   char *         options_string);

/**
 * @brief Turn a Rp2Mag struct into xml suitable for the Rio Pro 2 File Format
 * header.
 *
 * @param mag Pointer to the struct to turn into XML.
 * @param output[] Buffer to write the XML to. Must be at least
 * RP2_MAG_MAX_XML_LEN.
 */
void rp2mag_to_xml(const struct Rp2Mag *mag, char output[]);

/**
 * @brief Get the total number of whole bytes needed for all the encoded track
 * data.
 *
 * @param mag Pointer to the Rp2Mag struct containing the Mag data to use.
 *
 * @return The Sum of round bytes taken up by the encoded tracks. Useful for the
 * "<binary-size>" tag in an Rio Pro 2 PRN header.
 */
size_t rp2mag_total_encoded_as_bytes(const struct Rp2Mag *mag);

#endif /* RP2_MAG_H */
