/**
 * @file pr2comms.h
 *
 * @brief structures to support the job file format for the Rio Pro 2.
 *
 * @copyright Magicard Ltd.
 *
 ******************************************************************************/

#ifndef PR2COMMS_H
#define PR2COMMS_H

/******************************************************************************/

#include "optima_types.h"

/******************************************************************************/

/**
 * The printer capability flags type.
 *
 * A CAPABILITY_FLAGS value is formed by bitwise or-ing together the flags for
 * individual capabilities.  For example, a mag duplex printer would indicate a
 * CAPABILITY_FLAGS value of (CPF_ENCODE_MAG | CPF_PRINT_DUPLEX).
 */

/** CAPABILITY_FLAGS value indicating a magnetic encoding module is fitted.*/
#define CAP_ENCODE_MAG 0x00000001

/** CAPABILITY_FLAGS value indicating that duplex printing is enabled.*/
#define CAP_PRINT_DUPLEX 0x00000002

/** CAPABILITY_FLAGS value indicating that a chip encoding module is fitted.*/
#define CAP_ENCODE_CHIP 0x00000010

/** PARAM_ID value for wireless network information.*/
#define PID_RO_WIFI 50

/** PARAM_ID value for wireless disconnect.*/
#define PID_RO_WIFISERVERCONNECT 51

/** PARAM_ID value for wireless connect.*/
#define PID_RO_WIFISERVERDISCONNECT 52

/** PARAM_ID value for wireless network information.*/
#define PID_RO_WIFICONNECTSTATUS 53

/** PARAM_ID value for wireless network list.*/
#define PID_RO_WIFINETWORKSLIST 54

/** The maximum size of the version field (including the null terminator).*/
#define MAX_VERSION 32

/** The maximum size of the model name field (including the null terminator).*/
#define MAX_MODEL_NAME 32

/** The maximum size of the serial number field (inc null).*/
#define MAX_SERIAL_NUM 32

/** The maximum size of the manufacturer field (inc null).*/
#define MAX_MANF_NAME 32

/**
 * The PR360_RO_IDENTITY payload structure.
 */
typedef struct {
    /** The printer firmware version. */
    uint8 id_firmware_version[MAX_VERSION];

    /** The printer serial number. */
    uint8 id_printer_serial[MAX_SERIAL_NUM];

    /** The main PCB serial number. */
    uint8 id_main_board_serial[MAX_SERIAL_NUM];

    /** The printhead serial number. */
    uint8 id_printhead_serial[MAX_SERIAL_NUM];

    /** The SD card serial number (CID). */
    uint8 id_sd_card_serial[MAX_SERIAL_NUM];

    /** The magnetic encoder serial number. */
    uint8 id_mag_encoder_serial[MAX_SERIAL_NUM];

    /** The printer's full model name. */
    uint8 id_model_name[MAX_MODEL_NAME];

    /** The printer's dealer code. */
    uint32 printer_dealer_code;

    /** The printer's partner code. */
    uint32 printer_partner_code;

    /** The printer's capabilities. */
    CAPABILITY_FLAGS id_capability_flags;

    /** The current firmware's size, in bytes. */
    uint32 firmware_size;

    /** The manufacturer. */
    uint8 id_manufacturer[MAX_MANF_NAME];

    /** The total number of cards printed on this printer. */
    uint32 stats_total_cards;

    /** The total number of panels printed on this printer. */
    uint32 stats_total_panels;

    /** The total number of cards printed with the current printhead. */
    uint32 stats_cards_on_printhead;

    /** The total number of panels printed with the current printhead. */
    uint32 stats_panels_on_printhead;

    /** The number of cards printed since the last clean. */
    uint32 stats_cards_since_clean;

    /** The number of panels printed since the last clean. */
    uint32 stats_panels_since_clean;

    /** The number of cleaning cycles performed. */
    uint32 stats_cleaning_cycles_performed;

    /** The part number of the currently fitted colour film. */
    uint8 dye_name[32];

    /** The serial number of the currently fitted colour film. */
    uint8 dye_serial[32];

    /**The dye ribbon manufacturer code. */
    uint32 dye_ribbon_manufacturer;

    /** The total prints available with the currently fitted colour film. */
    uint32 dye_total_prints;

    /** The number of prints remaining on the currently fitted colour film. */
    uint32 dye_prints_remaining;

    /** The colour panel length in millimetres, or zero if the film has no such
     * panel. */
    uint32 dye_ymc_panel_length;

    /** The first black panel length in millimetres, or zero if the film has no
     * such panel. */
    uint32 dye_black1_panel_length;

    /** The second black panel length in millimetres, or zero if the film has no
     * such panel. */
    uint32 dye_black2_panel_length;

    /** The overcoat panel length in millimetres, or zero if the film has no
     * such panel. */
    uint32 dye_overcoat_panel_length;

    /** The number of cards since the last clean at which a cleaning prompt is
     * triggered. */
    uint32 clean_cards_between_prompts;

    /** Non-zero if cleaning is overdue, zero otherwise. */
    uint32 clean_overdue;

    uint8 operation_hand_feed[32];

    /** The UV panel length in millimetres, or zero if the film has no such
     * panel. */
    uint32 colour_uv_panel_length;

    /** The colour film's dealer code. */
    uint32 colour_dealer_code;

    /** The colour film's partner code. */
    uint32 colour_partner_code;

    /** The transfer film's dealer code. */
    uint32 transfer_dealer_code;

    /** The transfer film's partner code. */
    uint32 transfer_partner_code;

    /** Bits 23-0 of this are bits 47--24 of the MAC address.  Bits 31-24 are
     * set to zero. */
    uint32 mac_hi;

    /** Bits 23-0 of this are bits 23--0 of the MAC address.  Bits 31-24 are set
     * to zero. */
    uint32 mac_lo;

    /** The IP address obtained from DHCP. */
    uint32 dhcp_ip;

    /** The default gateway obtained from DHCP. */
    uint32 dhcp_gateway;

    /** The subnet mask obtained from DHCP. */
    uint32 dhcp_subnet;

    /** Status Word indicating comms status*/
    uint32 communication_status;

} PR360_RO_STATUS;

/******************************************************************************/

/*
 * PR360_RO_ERROR definitions.
 */

/**
 * The PR360_RO_ERROR payload structure.
 *
 * The ERROR_CODE type and valid error codes values are defined in the shared
 * header error_codes.h.
 */
typedef struct {
    /** The printer's current error code. */
    ERROR_CODE error_code;

    /** The error context code */
    uint32 error_context;

} PR360_RO_ERROR;

/**
 * The PR360_RO_IDENTITY payload structure.
 */
typedef struct {
    /** The printers ip address. */
    uint8 net_ip[16];

    /** The printer gateway address. */
    uint8 net_gateway[16];

    /** The printer subnet address. */
    uint8 net_subnet[16];

    /** 0 if static address in use. */
    uint8 use_dhcp;

    /** 5 - 5250000 */
    uint32 power_off_timeout;

} PR360_RO_TOOLQUERY;

/**
 * The PR360_RO_WIRELESSQUERY payload structure.
 */
typedef struct {
    /** the list of network interfaces */
    uint8 net_interfaces_list[1024];

    /** The printer wifi ip address. */
    uint8 wifi_ip[16];

    /** The printer wifi mac address. */
    uint8 wifi_mac[18];

    /** The printer wifi subnet address. */
    uint8 wifi_subnet[16];

    /** The printer net gateway address. */
    uint8 net_gateway[16];

    /** The printers connected wifi network */
    uint8 wifi_network[256];

    /** The printers discovered wifi networks list */
    uint8 wifi_networks_list[1024];

    /** The printer wifi static-gateway address. */
    uint8 wifi_static_gateway[16];

    /** The printer wifi static-ip address. */
    uint8 wifi_static_ip[16];

    /** The printers static wifi name server 1 */
    uint8 wifi_static_nameserver_1[256];

    /** The printers static wifi name server 2 */
    uint8 wifi_static_nameserver_2[256];

    /** The printer wifi static-subnet address. */
    uint8 wifi_static_subnet[16];

    /** 0 if static wifi address in use. */
    uint8 wifi_use_dhcp;

    /** 1 if wifi enabled, 2 disabled. */
    uint8 wifi_enabled;

} PR360_RO_WIFIQUERY;

/** Flag is set when the USB comms are active */
#define STATUS_USB_ACTIVE 0x00000001

/** Flag is set when Ethernet is active post DHCP IP Address assignment*/
#define STATUS_ETH_ACTIVE 0x00000002

/** Flag is set when the printer is busy upgrading*/
#define STATUS_BUSY 0x00000004

/*
 * JOB_GET_HOLOKOTE_PREVIEWS definitions.
 */

/* No extra data required beyond the header. */

/** The number of HoloKote slots available. */
#define NUM_HOLOKOTES 10

/** The length of the unique key associated with a HoloKote watermark. */
#define HOLOKOTE_UNIQUE_KEY_LEN 256

/** The width of a HoloKote preview image, in pixels. */
#define HOLOKOTE_PREVIEW_WIDTH (WIDTH_300DPI / 4)

/** The height of a HoloKote preview image, in pixels. */
#define HOLOKOTE_PREVIEW_HEIGHT (HEIGHT_300DPI / 4)

/** The size of a 1 bit per pixel HoloKote thumbnail image, in bytes.
 * Padded to 32-bit boundary.  */
#define HOLOKOTE_PREVIEW_SIZE_1BPP                                             \
    ((((HOLOKOTE_PREVIEW_WIDTH * HOLOKOTE_PREVIEW_HEIGHT) + 31) & ~31) / 8)

/** The size of a 16 bits per pixel HoloKote thumbnail image, in bytes. */
#define HOLOKOTE_PREVIEW_SIZE_16BPP                                            \
    (HOLOKOTE_PREVIEW_WIDTH * HOLOKOTE_PREVIEW_HEIGHT * 2)

#define HOLOKOTE_PREVIEW_SIZE_HEADER 256

#endif /* PR2COMMS_H */
