/***************************************************************************/
/**                                                                       **/
/**                            MAGICARD LTD                               **/
/**                                                                       **/
/***************************************************************************/
/**                                                                       **/
/**  PROJECT      : Magicard Driver                                       **/
/**                                                                       **/
/**  MODULE NAME  : MagEncd.C                                             **/
/**                                                                       **/
/**  SIN No.      : 2076                                                  **/
/**                                                                       **/
/**  COPYRIGHT    : Magicard Ltd				                          **/
/**                                                                       **/
/***************************************************************************/

#include <ctype.h>
#include <cups/cups.h>
#include <cups/ppd.h>
#include <cups/raster.h>
#include <fcntl.h>
#include <stdlib.h>

#include "debug_utils.h"
#include "magencd.h"
#include "magigen.h"
#include "utils.h"

/* Static Functions */
static size_t LocateStartSentinelPosition(LPMAGTRACKINFO lpTrackInfo);

/* Static Variables */
static bool bFoundSpecifier    = false;
static bool bFoundSpecifierEnd = false;
static bool bFirstCommaFound   = false;
static int  iMagControl        = 0;
static int  iConcTrack         = MAG_TRACK_NONE;

#ifndef FROM_ULTRA
static char ValidCommands[][MAX_COMMAND_SIZE] = {
        "BPI", "MPC", "COE", "GEM", "CHP", "EJT", "CHIP", JIS2_COMMAND};

/*
 * Valid Character Table for Tracks 1, 2 and 3
 *
 * The table is defined from MIN_CHAR_TRACKn to MAX_CHAR_TRACKn
 *      1 = If the character can be used between the SS and the ES.
 *      0 = If the character can NOT be used between the SS and the ES.
 *  Please note that the ES itself is defined as 1
 *  Relate the table with MAGTRACKINFO, at InitializeMagneticEncodeTrackInfo()
 */

// Allow all characters to be valid between start and end sentinel (SPR 20092
// part 2)

static bool ValidCharCodesISO_Track1[] = {
        //  sp  !   "   #   $   %   &   '   (   )   *   +   ,   -   .   /
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,

        //  0   1   2   3   4   5   6   7   8   9   :   ;   <   =   >   ?
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,

        //  @   A   B   C   D   E   F   G   H   I   J   K   L   M   N   O
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,

        //  P   Q   R   S   T   U   V   W   X   Y   Z   [   \   ]   ^   _
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1
};

static bool ValidCharCodesJIS_Track1[] = {
        //  sp  !   "   #   $   %   &   '   (   )   *   +   ,   -   .   /
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,

        //  0   1   2   3   4   5   6   7   8   9   :   ;   <   =   >   ?
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,

        //  @   A   B   C   D   E   F   G   H   I   J   K   L   M   N   O
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,

        //  P   Q   R   S   T   U   V   W   X   Y   Z   [   \   ]   ^   _
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,

        //  '   a   b   c   d   e   f   g   h   i   j   k   l   m   n   o
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,

        //  p   q   r   s   t   u   v   w   x   y   z   {   |   }   ~   DEL
            1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1
};
#endif // NOT FROM_ULTRA

/******************************************************************************
 *  InitializeMagneticEncodeTrackInfo()
 *      Initialize MAGTRACKINFO structure.
 *
 *  Returns:
 *      None
 *****************************************************************************/

void InitializeMagneticEncodeTrackInfo(PDEVDATA          pdev,
                                       struct settings_ *settings) {
    bool AutoInsert;

    memset(&pdev->magTrackInfo, 0, sizeof(MAGTRACKINFO) * MAX_TRACK_NUMBER);
    // Mag Tracks 0 & 4 are special tracks for passing of commands by an
    // application to a
    // printer via the header, 0 = front side, 4 = back side
    pdev->magTrackInfo[MAG_TRACK_INDEX0].ulMaxTrackLen = MAX_NUMCHAR_TRACK0;
    pdev->magTrackInfo[MAG_TRACK_INDEX0].wEndSentinel  = ISO_ENDSENTINEL;
    pdev->magTrackInfo[MAG_TRACK_INDEX0].bSSAutoInsert = false;
    pdev->magTrackInfo[MAG_TRACK_INDEX0].bESAutoInsert = false;
    pdev->magTrackInfo[MAG_TRACK_INDEX0].wMinCharCode  = MIN_CHAR_TRACK0;
    pdev->magTrackInfo[MAG_TRACK_INDEX0].wMaxCharCode  = MAX_CHAR_TRACK0;
    pdev->magTrackInfo[MAG_TRACK_INDEX0].pValidCharList =
            ValidCharCodesJIS_Track1;

    pdev->magTrackInfo[MAG_TRACK_INDEX4].ulMaxTrackLen = MAX_NUMCHAR_TRACK4;
    pdev->magTrackInfo[MAG_TRACK_INDEX4].wEndSentinel  = ISO_ENDSENTINEL;
    pdev->magTrackInfo[MAG_TRACK_INDEX4].bSSAutoInsert = false;
    pdev->magTrackInfo[MAG_TRACK_INDEX4].bESAutoInsert = false;
    pdev->magTrackInfo[MAG_TRACK_INDEX4].wMinCharCode  = MIN_CHAR_TRACK0;
    pdev->magTrackInfo[MAG_TRACK_INDEX4].wMaxCharCode  = MAX_CHAR_TRACK0;
    pdev->magTrackInfo[MAG_TRACK_INDEX4].pValidCharList =
            ValidCharCodesJIS_Track1;

    // Initialize members which will be different between each track.
    pdev->magTrackInfo[MAG_TRACK_INDEX1].wStartSentinel =
            ISO_STARTSENTINEL_TRACK1;
    pdev->magTrackInfo[MAG_TRACK_INDEX1].wEndSentinel  = ISO_ENDSENTINEL;
    pdev->magTrackInfo[MAG_TRACK_INDEX1].ulMaxTrackLen = MAX_NUMCHAR_TRACK1;
    pdev->magTrackInfo[MAG_TRACK_INDEX1].wMinCharCode  = MIN_CHAR_TRACK1;
    if (settings->bJIS2Enabled) {
        pdev->magTrackInfo[MAG_TRACK_INDEX1].wMaxCharCode = MAX_CHAR_ISO_TRACK1;
        pdev->magTrackInfo[MAG_TRACK_INDEX1].pValidCharList =
                ValidCharCodesISO_Track1;
    } else {
        pdev->magTrackInfo[MAG_TRACK_INDEX1].wMaxCharCode = MAX_CHAR_JIS_TRACK1;
        pdev->magTrackInfo[MAG_TRACK_INDEX1].pValidCharList =
                ValidCharCodesJIS_Track1;
    }
    pdev->magTrackInfo[MAG_TRACK_INDEX1].wBitsPerChar =
            settings->nBitsPerChar_Track1;
    pdev->magTrackInfo[MAG_TRACK_INDEX1].wBitsPerInch =
            (settings->nBitsPerInch_Track1) ? 75 : 210;

    pdev->magTrackInfo[MAG_TRACK_INDEX1].wCoercivity =
            settings->nCoercivity; // altered if set in passed string 0=COEH
                                   // 1=COEL
    pdev->magTrackInfo[MAG_TRACK_INDEX2].wCoercivity = settings->nCoercivity;
    pdev->magTrackInfo[MAG_TRACK_INDEX3].wCoercivity = settings->nCoercivity;

    pdev->magTrackInfo[MAG_TRACK_INDEX2].wStartSentinel =
            ISO_STARTSENTINEL_TRACK2;
    pdev->magTrackInfo[MAG_TRACK_INDEX2].wEndSentinel  = ISO_ENDSENTINEL;
    pdev->magTrackInfo[MAG_TRACK_INDEX2].wMinCharCode  = MIN_CHAR_TRACK1;
    pdev->magTrackInfo[MAG_TRACK_INDEX2].wMaxCharCode  = MAX_CHAR_ISO_TRACK1;
    pdev->magTrackInfo[MAG_TRACK_INDEX2].ulMaxTrackLen = MAX_NUMCHAR_TRACK1;
    pdev->magTrackInfo[MAG_TRACK_INDEX2].pValidCharList =
            ValidCharCodesISO_Track1;
    pdev->magTrackInfo[MAG_TRACK_INDEX2].wBitsPerChar =
            settings->nBitsPerChar_Track2;
    pdev->magTrackInfo[MAG_TRACK_INDEX2].wBitsPerInch =
            (settings->nBitsPerInch_Track2) ? 75 : 210;

    pdev->magTrackInfo[MAG_TRACK_INDEX3].wStartSentinel =
            ISO_STARTSENTINEL_TRACK3;
    pdev->magTrackInfo[MAG_TRACK_INDEX3].wEndSentinel  = ISO_ENDSENTINEL;
    pdev->magTrackInfo[MAG_TRACK_INDEX3].wMinCharCode  = MIN_CHAR_TRACK1;
    pdev->magTrackInfo[MAG_TRACK_INDEX3].wMaxCharCode  = MAX_CHAR_ISO_TRACK1;
    pdev->magTrackInfo[MAG_TRACK_INDEX3].ulMaxTrackLen = MAX_NUMCHAR_TRACK1;
    pdev->magTrackInfo[MAG_TRACK_INDEX3].pValidCharList =
            ValidCharCodesISO_Track1;
    pdev->magTrackInfo[MAG_TRACK_INDEX3].wBitsPerChar =
            settings->nBitsPerChar_Track3;
    pdev->magTrackInfo[MAG_TRACK_INDEX3].wBitsPerInch =
            (settings->nBitsPerInch_Track3) ? 75 : 210;

    if (HELIX_OEM(settings) || PRO360_OEM(settings)) {
        AutoInsert =
                (settings->nTrackOptions) ? true : false;

        pdev->magTrackInfo[MAG_TRACK_INDEX1].bSSAutoInsert =
                (settings->nBitsPerChar_Track1 > 1) ? 0 : AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX1].bESAutoInsert =
                (settings->nBitsPerChar_Track1 > 1) ? 0 : AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX2].bSSAutoInsert =
                (settings->nBitsPerChar_Track2 > 1) ? 0 : AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX2].bESAutoInsert =
                (settings->nBitsPerChar_Track2 > 1) ? 0 : AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX3].bSSAutoInsert =
                (settings->nBitsPerChar_Track3 > 1) ? 0 : AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX3].bESAutoInsert =
                (settings->nBitsPerChar_Track3 > 1) ? 0 : AutoInsert;

        pdev->magTrackInfo[MAG_TRACK_INDEX1].bHexData =
                (settings->nBitsPerChar_Track1 == 3) ? true : false;
        pdev->magTrackInfo[MAG_TRACK_INDEX2].bHexData =
                (settings->nBitsPerChar_Track2 == 3) ? true : false;
        pdev->magTrackInfo[MAG_TRACK_INDEX3].bHexData =
                (settings->nBitsPerChar_Track3 == 3) ? true : false;
    } else {
        if (RIO_OEM(settings) || ENDURO_OEM(settings)) {
            AutoInsert = (settings->nTrackOptions) ?
                                 true :
                                 false;

        } else {
            AutoInsert = true;
        }

        pdev->magTrackInfo[MAG_TRACK_INDEX1].bSSAutoInsert = AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX1].bESAutoInsert = AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX2].bSSAutoInsert = AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX2].bESAutoInsert = AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX3].bSSAutoInsert = AutoInsert;
        pdev->magTrackInfo[MAG_TRACK_INDEX3].bESAutoInsert = AutoInsert;
    }

    bFoundSpecifier    = false;
    bFoundSpecifierEnd = false;
    bFirstCommaFound   = false;
    iMagControl        = 0;
    iConcTrack         = MAG_TRACK_NONE;
}

/******************************************************************************
 *  ValidMagDataChar()
 *      Checks if the given character is in the valid mag data character list
 *
 *  Returns:
 *      bool = true if character is in the list
 *****************************************************************************/

bool ValidMagDataChar(LPMAGTRACKINFO lpTrackInfo, uint8_t CharIn) {
    bool result = false;

    if (CharIn >= lpTrackInfo->wMinCharCode &&
        CharIn <= lpTrackInfo->wMaxCharCode) {
        if (lpTrackInfo->pValidCharList[((int)CharIn & 0xff) -
                                        lpTrackInfo->wMinCharCode]) {
            result = true;
        }
    }
    // check for hex chars
    if (result == false) {
        uint8_t cmpstr[2];
        cmpstr[0] = (uint8_t)CharIn;
        cmpstr[1] = 0;
        if (strtok((char *)&cmpstr, "0123456789ABCDEFabcdef") == NULL)
            result = true;
    }
    return result;
}

/******************************************************************************
 *  JIS2CommandInString()
 *      Determines if a JIS2 Command is held in the mag encoding string
 *
 *  Returns:
 *      true or false
 *****************************************************************************/

bool JIS2CommandInString(const MAGTRACKINFO *lpTrackInfo) {
   TRACE_IN;
   TRACE("Looking for \"%s\" in %s", JIS2_COMMAND, lpTrackInfo->TrackData);
    // Test that lpTrackInfo is not empty
    if (lpTrackInfo == NULL) {
        return false;
    }

    // Check if there are any embedded commands in the string
    if (strrchr(lpTrackInfo->TrackData, CHAR_COMMA) == NULL) {
        return false;
    }

    // Is there a JIS2 command is the string
    if (strstr(lpTrackInfo->TrackData, JIS2_COMMAND) != NULL) {
        return true;
    }

    return false;
}

/******************************************************************************
 *  StartSentinel()
 *      Returns the Start Sentinel character for the selected track and encoding
 *      method
 *
 *  Returns:
 *      End Sentinel character
 *****************************************************************************/
int StartSentinel(LPMAGTRACKINFO lpTrackInfo, struct settings_ *settings) {
    if (settings->bJIS2Enabled || JIS2CommandInString(lpTrackInfo)) {
        return JIS_STARTSENTINEL;
    } else {
        return lpTrackInfo->wStartSentinel;
    }
}

/******************************************************************************
 *  EndSentinel()
 *      Returns the End Sentinel character for the selected encoding method
 *
 *  Returns:
 *      End Sentinel character
 *****************************************************************************/
int EndSentinel(LPMAGTRACKINFO lpTrackInfo, struct settings_ *settings) {
    if (settings->bJIS2Enabled || JIS2CommandInString(lpTrackInfo)) {
        return JIS_ENDSENTINEL;
    } else {
        return ISO_ENDSENTINEL;
    }
}
/*****************************************************************************/
void PerformPostProcessing(LPMAGTRACKINFO    lpTrackInfo,
                           struct settings_  *settings) {
   TRACE_IN;
   int nTrackLen = strlen(lpTrackInfo->TrackData);
   TRACE("nTrackLen = %d", nTrackLen);

   // Is the last character a space in the string?
   // If so remove it as it's more than likely not needed
   if (nTrackLen) {
      if (lpTrackInfo->TrackData[nTrackLen - 1] == ' ') {
         lpTrackInfo->TrackData[nTrackLen - 1] = '\0';
      }
   }

   // if Helix search for MPC or BPI entries
   TRACE("settings->OEM = %d", settings->OEM);
   if (settings->OEM == OEM_HELIX || settings->OEM == OEM_PRO360) {
      char *   pFoundStr = NULL;
      char *   pStartStr = NULL;
      uint32_t i = 0;
      uint16_t bpi, bpc, trunc = 7;
      uint32_t ulTrackLen = strlen(lpTrackInfo->TrackData);
      TRACE("ulTrackLen (again?) = %d", ulTrackLen);

      if (ulTrackLen < 10) {
         WARN_STR("ulTrackLen too short - not encoding.");
         TRACE_OUT;
         return;
      }

      // determine if the input string contains BPI210
      bpi = 210;
      TRACE_STR("Searching for 210 BPI definition in encoding string...");
      pFoundStr = strstr(lpTrackInfo->TrackData, "BPI210,");
      if (!pFoundStr) {
         TRACE_STR("210 BPI definition not found.");
         bpi = 75;
         trunc = 6;
         TRACE_STR("Searching for 75 BPI definition in encoding string...");
         pFoundStr = strstr(lpTrackInfo->TrackData, "BPI75,");
         if (pFoundStr == NULL){
            TRACE_STR("75 BPI definition not found.");
         }
      }

      if (pFoundStr) {
         TRACE("Found BPI of %d", bpi);
         lpTrackInfo->wBitsPerInch = bpi;
         // now remove BPI210, from the string
         pStartStr = lpTrackInfo->TrackData;
         i         = (uint32_t)(pFoundStr - pStartStr);
         mc_strncpy((char *)&lpTrackInfo->TrackData[i],
               pFoundStr + trunc,
               ulTrackLen - trunc);
      }

      ulTrackLen = strlen(lpTrackInfo->TrackData);
      TRACE("ulTrackLen is now = %d", ulTrackLen);
      if (ulTrackLen < 10) {
         WARN_STR("ulTrackLen too short - stopping encoding.");
         TRACE_OUT;
         return;
      }

      TRACE_STR("BITS PER CHAR PROCESSING.");
      // determine if the input string contains MPC5
      bpc = 2;
      TRACE_STR("Looking in the track data for 2 bits per char.");
      pFoundStr = strstr(lpTrackInfo->TrackData, "MPC1,");
      if (!pFoundStr) {
         TRACE_STR("Not found. Looking for 3 bits per char.");
         bpc = 3;
         pFoundStr = strstr(lpTrackInfo->TrackData, "MPC4,");
      }

      if (!pFoundStr) {
         TRACE_STR("Not found.");
         bpc = 1;
         TRACE_STR("Looking for 1 bit per char.");
         pFoundStr = strstr(lpTrackInfo->TrackData, "MPC5,");
      }

      if (!pFoundStr) {
         TRACE_STR("Not found.");
         bpc = 0;
         TRACE_STR("Looking for 7 bit per char.");
         pFoundStr = strstr(lpTrackInfo->TrackData, "MPC7,");
      }

      if (pFoundStr) {
         TRACE("Found! Bits per char set to %d", bpc);
         lpTrackInfo->wBitsPerChar = bpc;
         if (bpc > 1) {
            // binary or hex; never insert sentinals
            lpTrackInfo->bESAutoInsert = lpTrackInfo->bSSAutoInsert = false;
         }

         // now remove MPCn, from the string
         pStartStr = lpTrackInfo->TrackData;
         i         = (uint32_t)(pFoundStr - pStartStr);
         mc_strncpy((char *)&lpTrackInfo->TrackData[i],
               pFoundStr + 5,
               ulTrackLen - 5);
      }

      ulTrackLen = strlen(lpTrackInfo->TrackData);
      TRACE("ulTrackLen is now = %d", ulTrackLen);
      if (ulTrackLen < 10) {
         WARN_STR("ulTrackLen too short - stopping encoding.");
         TRACE_OUT;
         return;
      }

      // determine if the input string contains COEH or COEL
      pFoundStr = NULL;

      TRACE_STR("COERCIVITY PROCESSING.");

      TRACE_STR("Looking for HiCo, with comma.");
      pFoundStr = strstr(lpTrackInfo->TrackData, "COEH,");
      if (pFoundStr == NULL) {
         TRACE_STR("Not found.");
         TRACE_STR("Looking for HiCo, with semi-colon.");
         pFoundStr = strstr(lpTrackInfo->TrackData, "COEH;");
      }

      if (pFoundStr) {
         TRACE_STR("HiCo found.");
         lpTrackInfo->wCoercivity = MAG_COERCIVITY_HIGH;
      }

      if (pFoundStr == NULL) {
         TRACE_STR("HiCo not found. Looking for LoCo + comma.");
         pFoundStr = strstr(lpTrackInfo->TrackData, "COEL,");
         if (pFoundStr == NULL) {
            TRACE_STR("Not found.");
            TRACE_STR("Looking for LoCo + semi-colon.");
            pFoundStr = strstr(lpTrackInfo->TrackData, "COEL;");
         }

         if (pFoundStr) {
            TRACE_STR("Found LoCo.");
            lpTrackInfo->wCoercivity = MAG_COERCIVITY_LOW;
         }
      }

      if (pFoundStr == NULL) {
         TRACE_STR("Not found.");
         TRACE_STR("Looking for MidCo + colon");
         pFoundStr = strstr(lpTrackInfo->TrackData, "COEM,");
         if (pFoundStr == NULL) {
            TRACE_STR("Not found.");
            TRACE_STR("Looing for MidCo + semi-colon.");
            pFoundStr = strstr(lpTrackInfo->TrackData, "COEM;");
         }

         if (pFoundStr) {
            TRACE_STR("MidCo found.");
            lpTrackInfo->wCoercivity = MAG_COERCIVITY_MEDIUM;
         }
      }

      if (pFoundStr) {
         // now remove COEH or COEL, from the string
         pStartStr = lpTrackInfo->TrackData;
         i = (uint32_t)(pFoundStr - pStartStr);
         mc_strncpy((char *)&lpTrackInfo->TrackData[i],
               pFoundStr + 5,
               ulTrackLen - 5);
      }

   }

   TRACE("After all processing, TrackData is [%s]", lpTrackInfo->TrackData);

   // Perform any required post-processing of the magnetic encoding string
   if (!lpTrackInfo->bSSAutoInsert) {
      // If we are not required to automatically insert a start sentinel
      // assume that we have found one, then one will not get inserted.
      lpTrackInfo->bFoundSS = true;
   }

   if (!lpTrackInfo->bESAutoInsert) {
      // If we are not required to automatically insert an end sentinel
      // assume that we have found one, then one will not get inserted.
      lpTrackInfo->bFoundES = true;
   }

   if (lpTrackInfo->bFoundSS && lpTrackInfo->bFoundES) {
      // We are finished recording the track
      TRACE_STR("Setting bComplete to true.");
      lpTrackInfo->bComplete = true;
   } else if (lpTrackInfo->bFoundSS) {
      // Although a start sentinel was found, the end sentinel was not.
      // This is either because a buffer limit was reached, or the end
      // sentinel
      // was missing from the test case. Either way an end sentinel must be
      // added.

      uint32_t ulESPos =
         (uint32_t)strlen(lpTrackInfo->TrackData); // Default to add ES
      // after the last
      // recorded character

      if (lpTrackInfo->bBufferFull) {
         ulESPos--; // Overwrite the last recorded character with an end
         // sentinel
      }

      lpTrackInfo->TrackData[ulESPos] =
         (char)(EndSentinel(lpTrackInfo, settings) & 0xff);

      // We are finished recording the track
      TRACE_STR("Setting bComplete to true.");
      lpTrackInfo->bComplete = true;
   } else {
      // No start sentinel found - (end sentinel may or may not have been
      // seen but will be dealt with automatically)
      size_t ulSSPos = 0;
      size_t ulESPos = 0;
      size_t i       = 0;

      /*
       * The start sentinel is missing. This is due to one of the following...
       *  1) The SS was missing in the string provided by the application
       *  2) There is no track data - just commands
       * Now call LocateSSPosition() which parses the current string to find
       * a boundary between track commands and track data (if this exists).
       */
      ulSSPos = LocateStartSentinelPosition(lpTrackInfo);

      if (ulSSPos > 0) {
         // Valid track data was found

         // Calculate a new position for the end sentinel
         ulESPos = strlen(lpTrackInfo->TrackData);
         if (!lpTrackInfo->bFoundES) {
            ulESPos++;
         }

         // Ensure that we do not overrun the buffer length limit
         if (ulESPos >= MAG_MAXBUFFERNUM) {
            ulESPos = MAG_MAXBUFFERNUM - 1;
         }

         // Ensure that we do not overrun the track data limit
         if ((ulESPos - ulSSPos) > lpTrackInfo->ulMaxTrackLen) {
            ulESPos = ulSSPos + lpTrackInfo->ulMaxTrackLen - 1;
            lpTrackInfo->TrackData[ulESPos + 1] = (char)(0x00);
         }

         // Record the length of the track data
         lpTrackInfo->ulTrackLength = ulESPos - ulSSPos;

         // Move all track data across by one position
         for (i = ulESPos - 1; i > ulSSPos; i--) {
            lpTrackInfo->TrackData[i] = lpTrackInfo->TrackData[i - 1];
         }

         // Add in the start and end sentinel characters to complete the
         // magnetic encoding string
         lpTrackInfo->TrackData[ulSSPos] =
            (char)(StartSentinel(lpTrackInfo, settings) & 0xff);
         lpTrackInfo->TrackData[ulESPos] =
            (char)(EndSentinel(lpTrackInfo, settings) & 0xff);
      } else {
         // No track data was found, so do not add a start sentinel
      }
      // We are finished recording the track
      TRACE_STR("Setting bComplete to true.");
      lpTrackInfo->bComplete = true;
   }

   // Ready to handle another track
   iMagControl        = 0;
   bFoundSpecifier    = false;
   bFoundSpecifierEnd = false;
   TRACE_OUT;
}

/******************************************************************************
 *  RecordMagneticEncodeTrackInfo()
 *      Hook DDI TextOut call and search magnetic encoding data.
 *
 *  Returns:
 *      GCB_SUPPORTED if callback handled, GCB_NOT_SUPPORTED if not
 *****************************************************************************/

int RecordMagneticEncodeTrackInfo(PDEVDATA          pdev,
                                  char *            lpStrIn,
                                  struct settings_ *settings) {

    LPMAGTRACKINFO lpTrackInfo = NULL;
    uint8_t *      lpwTrans    = NULL;
    uint8_t        CharIn      = 0;
    int            iNumChar, cGlyphs = strlen(lpStrIn);
    int            ichar  = 0;
    int            iCount = 0;
    bool           bRet   = false;
    bool           bFoundTilde =
            false; // true = Found a tilde as the first character in the string
    bool bFoundTrack =
            false; // true = Found a valid track number as the second character
    int     iTrack = MAG_TRACK_NONE; // Track index track being processed
    int     i;
    int32_t CurrentTextY = -1;
    TRACE_IN;

    TRACE("cGlyphs:%u", cGlyphs);

    /*
     * Please note that we start with the assumption that text will be
     * printed as usual (without extracting it as magnetic encoding data).
     * If magnetic encoding data is found, bRet will be set to true
     * This prevents the text from being drawn.
     */

    /*
     * Please note that applications may split strings up into a series of
     * smaller strings
     * (down to individual characters) - but we are only interested in completed
     * strings.
     * The driver expects strings to be at least long enough to contain "~1,",
     * "~2,", or "~3,"
     * so now we will disallow any strings that are too short, and validate the
     * beginning
     * characters of the string
     */

    // Grab the current y coordinate of the string being passed so we close the
    // track
    // if no glyph position passed do not use it..

    // Check if the string begins with tilde.
    if (!settings->bConcatenateMagStrings) {
       TRACE_AT;
        if (cGlyphs < 3) {
            // Do not process the string in this case - (it will be printed as
            // normal)
            bRet = false;
            goto FINISH_RECORD;
        }
        // added check as to whether tilde is the first character passed in the
        // string..
        if (ichar == 0 && lpStrIn[ichar++] == CHAR_TILDE) {
            // Flag that a tilde has been found
            bFoundTilde = true;
            TRACE_STR("Tilde~ Mag Specifier found...");
            // Fall through so that other characters can be processed below.
        } else {
            // This string is not magnetic encoding data so do not process it.
            bRet = false;
            goto FINISH_RECORD;
        }

        // Check for the character immediately following the tilde
        if (bFoundTilde) {
            // Obtain and test for a valid track number in the string provided.
            CharIn = lpStrIn[ichar++];
            TRACE("CharIn: %c", CharIn);
            // If the character after the track number is a comma, ignore it as
            // one will be inserted anyway
            if (lpStrIn[ichar] == CHAR_COMMA) {
                ichar++;
            }

            switch (CharIn) {
                case INDEX_CHAR_TRACK0:
                    iTrack = MAG_TRACK_INDEX0;
                    break;
                case INDEX_CHAR_TRACK1:
                    iTrack = MAG_TRACK_INDEX1;
                    break;
                case INDEX_CHAR_TRACK2:
                    iTrack = MAG_TRACK_INDEX2;
                    break;
                case INDEX_CHAR_TRACK3:
                    iTrack = MAG_TRACK_INDEX3;
                    break;
                case INDEX_CHAR_TRACK4:
                    iTrack = MAG_TRACK_INDEX4;
                    break;

                default:
                    // WARNING : We could not find any valid track number.
                    iTrack = MAG_TRACK_NONE;
                    TRACE_STR("Printing string as normal");

                    // Do not process the string in this case - (it will be
                    // printed as normal)
                    bRet = false;
                    goto FINISH_RECORD;
                    break;
            }

            // Obtain track info structure for the appropriate track number
            lpTrackInfo = &pdev->magTrackInfo[iTrack];

            // Test whether there is already data recorded for this track on
            // this page
            if (lpTrackInfo->TrackData[0] != 0) {
                // We recorded this track earlier on within the page.
                // Do do not process the string further - (it will be printed as
                // normal)
                iTrack = MAG_TRACK_NONE;
                bRet   = false;
                goto FINISH_RECORD;
            } else {
                // Flag that we are processing a new track for this page
                TRACE_AT;
                bFoundTrack = true;

                // Clear the memory buffer for this track
                memset(lpTrackInfo->TrackData,
                       0,
                       sizeof(lpTrackInfo->TrackData));

                // Record the track number (and comma) in the output string.
                lpTrackInfo->TrackData[strlen(lpTrackInfo->TrackData)] =
                        (char)(CharIn & 0xff);
                lpTrackInfo->TrackData[strlen(lpTrackInfo->TrackData)] =
                        (char)(CHAR_COMMA & 0xff);

                TRACE("lpTrackInfo->TrackData = %s", lpTrackInfo->TrackData);

                // Process the string
                bRet = true;

                // Fall through so that other characters can be processed below.
            }
        }

        // Look for (and process) characters which follow after the tilde, track
        // number and comma
        if (bFoundTilde && bFoundTrack && iTrack != MAG_TRACK_NONE) {
           TRACE_AT;
            // Record the number of characters remaining in the hooked text
            iNumChar = cGlyphs - ichar;
            TRACE("iNumChar: %u", iNumChar);
            // The string is being processed by the mini-driver and will not be
            // printed
            bRet = true;

            for (iCount = 0; iCount < iNumChar; iCount++) {
                CharIn = lpStrIn[ichar++];
                TRACE("CharIn:%c", CharIn);
                // Check that this character is in the allowable range for this
                // track.
                if (ValidMagDataChar(lpTrackInfo, CharIn) || isupper(CharIn) ||
                    isdigit(CharIn) || (CharIn == ',')) {
                    // Store the valid character into the track data for this
                    // track
                    lpTrackInfo->TrackData[strlen(lpTrackInfo->TrackData)] =
                            (char)(CharIn & 0xff);

                    // Check for the start sentinel of this track
                    if (CharIn == StartSentinel(lpTrackInfo, settings)) {
                        // Flag that we found the SS so that we start counting
                        // track data characters
                        lpTrackInfo->bFoundSS = true;
                    }

                    // Check for the end sentinel of this track
                    if (CharIn == EndSentinel(lpTrackInfo, settings)) {
                        // Flag that we have found an ES
                        lpTrackInfo->bFoundES = true;

                        // Break out of the string construction loop
                        break;
                    }

                    if (lpTrackInfo->bFoundSS) {
                        // Increase the count of the track data length
                        lpTrackInfo->ulTrackLength++;

                        // Ensure that we do not overrun the track data limit
                        if (lpTrackInfo->ulTrackLength >=
                            lpTrackInfo->ulMaxTrackLen) {
                            /*
                             * WARNING: flag that we have hit the track data
                             * length limit
                             * and break out of the string construction loop
                             */
                            lpTrackInfo->bBufferFull = true;
                            break;
                        }
                    }

                    // Ensure that we do not overrun the overall track buffer
                    // limit (commands + track data)
                    if (strlen(lpTrackInfo->TrackData) >= MAG_MAXBUFFERNUM) {
                        /*
                         * WARNING: flag that we have hit the track buffer
                         * length limit
                         * and break out of the string construction loop
                         */
                        lpTrackInfo->bBufferFull = true;
                        break;
                    }
                }
            }
            TRACE("pdev->magTrackInfo[iTrack] = %s", lpTrackInfo->TrackData);
            TRACE("lpTrackInfo->TrackData = %s", lpTrackInfo->TrackData);
            // Perform any required post-processing of the magnetic encoding
            // string
            TRACE_STR("About to call post");
            PerformPostProcessing(lpTrackInfo, settings);
            TRACE_STR("returned from call post");
            TRACE("lpTrackInfo->TrackData = %s", lpTrackInfo->TrackData);

        }
    } else {
       TRACE_AT;
        //		+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
        //		For now, this is a duplication of the code used with tilde
        //    specifier.  This is in order to ensure that normal use of
        //    tilde for magnetic encoding is not compromised.
        //		+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

        // if we are recording the string and the next string passed is on a
        // different line close the track and start analysing the passed string
        if (iMagControl == 2) {
            // Obtain track info structure for the appropriate track number
            lpTrackInfo = &pdev->magTrackInfo[iConcTrack];
            PerformPostProcessing(lpTrackInfo, settings);
            bFirstCommaFound = false;
        }

        if (bFoundSpecifierEnd) {
            // Magnetic encoding data already sorted so do not process it.
            bRet = false;
            goto FINISH_RECORD;
        }

        while ((iCount++ < (int)cGlyphs) && !bFoundSpecifierEnd) {
            switch (iMagControl) {
                case 0:
                    // added check as to whether tilde is the first character
                    // passed in the string..
                    if (ichar == 0 && lpStrIn[ichar++] == CHAR_TILDE) {
                        TRACE_STR("Mag Specifier FOUND");
                        bFoundSpecifier    = true;
                        iMagControl        = 1;
                        pdev->LastMagTextY = CurrentTextY;
                    }
                    break;

                case 1:
                    CharIn = lpStrIn[ichar++];
                    switch (CharIn) {
                        case INDEX_CHAR_TRACK0:
                            iConcTrack = MAG_TRACK_INDEX0;
                            break;
                        case INDEX_CHAR_TRACK1:
                            iConcTrack = MAG_TRACK_INDEX1;
                            break;
                        case INDEX_CHAR_TRACK2:
                            iConcTrack = MAG_TRACK_INDEX2;
                            break;
                        case INDEX_CHAR_TRACK3:
                            iConcTrack = MAG_TRACK_INDEX3;
                            break;
                        case INDEX_CHAR_TRACK4:
                            iConcTrack = MAG_TRACK_INDEX4;
                            break;

                        default:
                            // WARNING : We could not find any valid track
                            // number.
                            iConcTrack = MAG_TRACK_NONE;

                            // Do not process the string in this case - (it will
                            // be printed as normal)
                            bRet = false;
                            goto FINISH_RECORD;
                            break;
                    }
                    bFirstCommaFound = false;
                    // Obtain track info structure for the appropriate track
                    // number
                    lpTrackInfo = &pdev->magTrackInfo[iConcTrack];

                    // Clear the memory buffer for this track
                    memset(lpTrackInfo->TrackData,
                           0,
                           sizeof(lpTrackInfo->TrackData));

                    // Record the track number (and comma) in the output string.
                    lpTrackInfo->TrackData[0] = (char)(CharIn & 0xff);
                    lpTrackInfo->TrackData[1] = (char)(CHAR_COMMA & 0xff);
                    iMagControl               = 2;
                    break;

                case 2:
                    if (lpStrIn[ichar] == CHAR_COMMA && !bFirstCommaFound) {
                        // Skip over the comma if one has been included
                        ichar++;
                        // mark the fact we've passed the first comma delimiter
                        // as we want the rest
                        bFirstCommaFound = true;
                    }

                    if (bFoundSpecifier && !bFoundSpecifierEnd &&
                        (iConcTrack != MAG_TRACK_NONE)) {
                        // Record the number of characters remaining in the
                        // hooked text
                        iNumChar = cGlyphs - ichar;

                        // The string is being processed by the mini-driver and
                        // will not be printed
                        bRet = true;

                        lpTrackInfo = &pdev->magTrackInfo[iConcTrack];

                        // Convert Unicode characters into ascii.
                        for (i = 0; i < iNumChar; i++) {
                            CharIn = lpStrIn[ichar++];

                            if (CharIn == CHAR_TILDE) {
                                bFoundSpecifierEnd = true;
                                break;
                            }

                            // Check that this character is in the allowable
                            // range for this track.
                            if (ValidMagDataChar(lpTrackInfo, CharIn) ||
                                isupper(CharIn) || isdigit(CharIn) ||
                                (CharIn == ',')) {
                                // Store the valid character into the track data
                                // for this track
                                lpTrackInfo->TrackData[strlen(
                                        lpTrackInfo->TrackData)] =
                                        (char)(CharIn & 0xff);

                                // Check for the start sentinel of this track
                                if (CharIn ==
                                    StartSentinel(lpTrackInfo, settings)) {
                                    // Flag that we found the SS so that we
                                    // start counting track data characters
                                    lpTrackInfo->bFoundSS = true;
                                }

                                // Check for the end sentinel of this track
                                if (CharIn ==
                                    EndSentinel(lpTrackInfo, settings)) {
                                    // Flag that we have found an ES
                                    lpTrackInfo->bFoundES = true;
                                    bFoundSpecifierEnd    = true;
                                    iMagControl           = 0;
                                }

                                if (lpTrackInfo->bFoundSS) {
                                    // Increase the count of the track data
                                    // length
                                    lpTrackInfo->ulTrackLength++;

                                    // Ensure that we do not overrun the track
                                    // data limit
                                    if (lpTrackInfo->ulTrackLength >=
                                        lpTrackInfo->ulMaxTrackLen) {
                                        /*
                                         * WARNING: flag that we have hit the
                                         * track data length limit
                                         * and break out of the string
                                         * construction loop
                                         */
                                        lpTrackInfo->bBufferFull = true;
                                        break;
                                    }
                                }

                                // Ensure that we do not overrun the overall
                                // track buffer limit (commands + track data)
                                if (strlen(lpTrackInfo->TrackData) >=
                                    MAG_MAXBUFFERNUM) {
                                    /*
                                     * WARNING: flag that we have hit the track
                                     * buffer length limit
                                     * and break out of the string construction
                                     * loop
                                     */
                                    lpTrackInfo->bBufferFull = true;
                                    break;
                                }
                            }
                        }
                    }
                    break;
            }
        }

        TRACE("bFoundSpecifier = %s", STR_BOOLEAN(bFoundSpecifier));
        TRACE("bFoundSpecifierEnd = %s", STR_BOOLEAN(bFoundSpecifierEnd));
        // Process the string only if a tilde has been sent
        if (bFoundSpecifier) {
            bRet = true;
        }

        if (bFoundSpecifierEnd) {
            PerformPostProcessing(lpTrackInfo, settings);
        }
    }
//	++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

FINISH_RECORD:

    // Free up any device data here
    if (lpwTrans != NULL) {
        free(lpwTrans);
    }

    TRACE("bRet = %s", STR_BOOLEAN(bRet));
    TRACE_OUT;
    return bRet;
}

/******************************************************************************
 *
 *  LocateStartSentinelPosition()
 *      Find the correct position for a start sentinel if it is missing from
 *      the string.This function does not modify the string in any way.
 *
 *  Params:
 *      lpTrackInfo    Pointer to the magnetic track information structure
 *
 *  Returns:
 *      size_t     Position of the start sentinal in the provided track info.
 *                 0 if a suitable position is not found.
 *
 *****************************************************************************/
static size_t LocateStartSentinelPosition(LPMAGTRACKINFO lpTrackInfo) {
    bool   bFoundCmd = true;
    char   strData[MAG_MAXBUFFERNUM]; // For copy of lpTrackInfo->TrackData
    size_t ulPos = 0;
    size_t i     = 0;
    size_t j     = 0;

    // Test that lpTrackInfo is not empty
    if (lpTrackInfo == NULL) {
        return 0;
    }

    // Make a local (upper-case) copy of lpTrackInfo->TrackData for processing
    mc_strncpy(strData, lpTrackInfo->TrackData, sizeof(strData));

    // Loop from the first comma to the last but one character
    for (i = 1; i < strlen(strData) - 1; i++) {
        // Locate a comma
        if (lpTrackInfo->TrackData[i] == CHAR_COMMA) {
            // Assume that the following command is invalid
            bFoundCmd = false;

            // Loop the list of valid commands for the track
            for (j = 0; j < sizeof(ValidCommands) / MAX_COMMAND_SIZE; j++) {
                // Compare the characters immediately following the comma
                // against a command
                if (strstr(strData + i + 1, ValidCommands[j]) ==
                    strData + i + 1) {
                    // Flag that a valid command was found, and break out of the
                    // inner loop
                    bFoundCmd = true;
                    break;
                }
            }
        }

        // If no valid command was found then we must have found the start
        // sentinel position
        if (!bFoundCmd) {
            // Record the current position and break out of the outer loop
            ulPos = i + 1;
            break;
        }
    }

    // Prevent start sentinel from being added to the very end of the string
    if (ulPos == strlen(strData)) {
        ulPos = 0;
    }

    return ulPos;
}
